"use client";
import { useEffect, useRef, useState } from "react";
import QRCode from "qrcode";
import { useLanguage } from "@/contexts/LanguageContext";

interface QRCodeGeneratorProps {
  url: string;
  size?: number;
  className?: string;
  dataAttribute?: string;
}

export default function QRCodeGenerator({
  url,
  size = 200,
  className = "",
  dataAttribute,
}: QRCodeGeneratorProps) {
  const { t } = useLanguage();
  const canvasRef = useRef<HTMLCanvasElement>(null);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    if (!canvasRef.current || !url) return;

    const generateQR = async () => {
      try {
        setError(null);
        await QRCode.toCanvas(canvasRef.current, url, {
          width: size,
          margin: 2,
          color: {
            dark: "#000000",
            light: "#FFFFFF",
          },
        });
      } catch (err) {
        console.error("QR Code generation error:", err);
        setError(t("common.qrCodeError") || "Failed to generate QR code");
      }
    };

    generateQR();
  }, [url, size, t]);

  if (error) {
    return (
      <div
        className={`flex items-center justify-center bg-muted rounded-lg ${className}`}
        style={{ width: size, height: size }}
      >
        <p className='text-sm text-muted-foreground text-center p-4'>{error}</p>
      </div>
    );
  }

  return (
    <div className={`inline-block ${className}`}>
      <canvas
        ref={canvasRef}
        className='rounded-lg border'
        style={{ width: size, height: size }}
        {...(dataAttribute && { [dataAttribute]: true })}
      />
    </div>
  );
}
